(function(ViewController, utils, autoSaveController, strings) {
    'use strict';

    var DEFAULT_CARD_STYLE =
            'position: relative;' +
            'box-sizing: border-box;' +
            'width: 100%;' +
            'max-width:400px;' +
            'color: #000;' +
            'text-decoration:none;';

    /**
     *
     * @constructor
     */
    function CardViewController() {
        ViewController.call(this);
    }

    /**
     * Creates a container element containing all the cards
     * elemements.
     *
     * @param {string} [containerStyle] - style values to be added as an attribute on the container element
     *
     * @return {HtmlElement}
     */
    function createContainerEl(style) {
        var containerEl = document.createElement('div');

        containerEl.classList.add(CARD_EL_CLASS_NAME);
        containerEl.classList.add(UNIFIED_CARD_EL_CLASS_NAME);
        containerEl.setAttribute('style', style || '');
        containerEl.setAttribute('contenteditable', 'false');

        return containerEl;
    }

    /**
     * Create the remove button element.
     *
     * @return {HtmlElement}
     */
    function createRemoveBtnEl() {
        var removeBtnEl = document.createElement('button');

        removeBtnEl.setAttribute('aria-label', strings.mailskd_accessibility_remove_attachment);

        removeBtnEl.classList.add(REMOVE_BTN_CLASS_NAME);
        // Styles to hide the remove button when viewing outside of a compose view
        // These should be overridden in the the css files for compose
        removeBtnEl.setAttribute('style',
                'position: absolute;' +
                'display: none;' +
                'height: 0;' +
                'width: 0;' +
                'border: none;' +
                'background: none;' +
                'color: transparent;'
            );

        return removeBtnEl;
    }

    /**
     * Create a Card Element
     *
     * @param {string} url - src url for cloud anchor element
     * @param {string} [style] - style values to be added as an attribute on the container element
     * @param {string} [containerStyle] - style values to be added as an attribute on the container element
     * @param {node} removeBtnEl - custom remove button
     *
     * @returns {object} map to created HtmlElements
     */
    function createCard(url, style, removeBtnEl) {
        var containerEl = createContainerEl(style);

        // if custom remove button is not set, create a new one
        if (!removeBtnEl) {
            removeBtnEl = createRemoveBtnEl();
            containerEl.appendChild(removeBtnEl);
        }

        return {
            containerEl: containerEl,
            removeBtnEl: removeBtnEl
        };
    }

    function findCardContainerEl(el) {
        while (el.parentElement) {
            if (el.parentElement.className && el.parentElement.className.indexOf(CARD_EL_CLASS_NAME) >= 0) {
                return el.parentElement;
            }
            el = el.parentElement;
        }
    }

    CardViewController.prototype = Object.create(ViewController.prototype);
    CardViewController.prototype.constructor = CardViewController;


    /**
     *
     * @param {object} properties
     * @param {string} [properties.url] - url used for wrapper anchor href
     * @param {array} [properties.classNames] - array of classes to add to the card container element
     * @param {string} [properties.style] - new styles to add to default card styles
     * @param {node} children - children nodes to insert into the card
     * @param {node} removeBtnEl - custom remove btn if needed
     *
     * @override
     */
    CardViewController.prototype.render = function(properties, children, removeBtnEl) {
        var renderFragment,
            card,
            classNames;

        properties = properties || {};

        card = createCard(
            properties.url || '',
            DEFAULT_CARD_STYLE + (properties.style || ''),
            removeBtnEl
        );

        // Insert children from parameter into the card
        card.containerEl.appendChild(children);

        // Add new class names from parameter
        classNames = properties.classNames || [];
        classNames.forEach(function(className) {
            card.containerEl.classList.add(className);
        });

        this.attach(card.containerEl, card.removeBtnEl);

        // Add a line break after the card and return the document fragment
        renderFragment = document.createDocumentFragment();
        renderFragment.appendChild(this.el);
        renderFragment.appendChild(document.createElement('br'));
        return renderFragment;
    };

    /**
     * Attach to an existing Attachment Card in the DOM
     *
     * @param {HtmlElement} cardEl
     * @param {HtmlElement} removeBtnEl
     * @override
     */
    CardViewController.prototype.attach = function(cardEl, removeBtnEl) {
        ViewController.prototype.attach.call(this, cardEl);
        this.removeBtnEl = removeBtnEl;

        /*
         * As the imageTapDeleteController handles all click events inside the compose body editor
         * using the element's tagName ('img') to bind logic, we must prevent the event from propogating.
         */
        this.el.addEventListener('click', function(ev) {
            var curEl = ev.target,
                container = findCardContainerEl(curEl),
                nextSibling;

            ev.stopPropagation();
            if (container) {
                if (!container.nextSibling || container.nextSibling.getAttribute && container.nextSibling.getAttribute("contenteditable") === "false") {
                    nextSibling = document.createElement("br");
                    container.parentElement && container.parentElement.insertBefore(nextSibling, container.nextSibling);
                } else {
                    nextSibling = container.nextSibling;
                }
            }
        });

        this.removeBtnEl.addEventListener('click', this.destroy.bind(this));
    };

    CardViewController.prototype.destroy = function() {
        this.removeBtnEl.removeEventListener('click', this.destroy);

        utils.removeElement(this.el);
        this.el = null;

        autoSaveController.handleInput(null);
    };

    CardViewController.prototype.createRemoveBtnEl = createRemoveBtnEl;

    window.CardViewController = CardViewController;
})(window.ViewController, window.utils, window.autoSaveController, window.strings);
